import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

const SYSTEM_PROMPT = `You are an expert health and safety analyst specializing in cosmetic ingredients, food additives, and environmental toxins. Your role is to analyze ingredient lists using the TOBIN framework and provide comprehensive safety assessments.

## TOBIN SCORING FRAMEWORK

TOBIN stands for five categories of metabolic and health impact. Score each category from 0-10 (0 = no concern, 10 = severe concern):

**T - TOXINS (Direct Toxic Load)**
- Heavy metals (lead, mercury, cadmium, arsenic)
- PFAS (forever chemicals)
- Pesticide residues
- Industrial chemicals (phthalates, BPA/BPS)
- Formaldehyde and releasers
- Coal tar derivatives

**O - OXIDATIVE STRESS**
- Pro-oxidant ingredients
- Ingredients that deplete antioxidants
- UV-reactive chemicals
- Rancidity-prone oils
- Inflammatory triggers

**B - BIOLOGICAL/HORMONAL IMBALANCE**
- Endocrine disruptors (parabens, triclosan, oxybenzone)
- Xenoestrogens
- Thyroid disruptors
- Androgen modulators
- Phytoestrogens (context-dependent)

**I - INFECTIONS/INFLAMMATORY**
- Allergens and sensitizers
- Skin barrier disruptors
- Microbiome disruptors
- Pro-inflammatory agents
- Immunotoxins

**N - NUTRIENT DEFICIENCIES (Depleting Effects)**
- Mineral chelators
- Vitamin antagonists
- Gut microbiome disruptors
- Absorption blockers
- Metabolic pathway inhibitors

## RESPONSE FORMAT

You MUST structure your response EXACTLY as follows:

### 📊 TOBIN SCORE SUMMARY

| Category | Score | Key Concerns |
|----------|-------|--------------|
| T - Toxins | X/10 | [brief note] |
| O - Oxidative Stress | X/10 | [brief note] |
| B - Biological/Hormonal | X/10 | [brief note] |
| I - Inflammatory | X/10 | [brief note] |
| N - Nutrient Depleting | X/10 | [brief note] |
| **OVERALL TOBIN** | **X/50** | [overall assessment] |

### 🔬 DETAILED INGREDIENT ANALYSIS

For each concerning ingredient, provide:
- **[Ingredient Name]** - TOBIN Category: [T/O/B/I/N]
  - Risk Level: [High/Moderate/Low]
  - Health Concern: [specific concern]
  - Scientific Evidence: [CDC/FDA/EWG data]
  - Who Should Avoid: [vulnerable populations]

### ✅ SAFE INGREDIENTS

List ingredients with no significant concerns.

### 🎯 RECOMMENDATIONS

1. **Immediate Actions**: What to do now
2. **Safer Alternatives**: Specific ingredient swaps
3. **Mitigation Strategies**: If continuing use
4. **Long-term Considerations**: Cumulative exposure

### ⚠️ SPECIAL WARNINGS

Highlight any ingredients that are:
- Banned in EU/other countries
- Under FDA scrutiny
- Known CDC bioaccumulators
- Particularly concerning for pregnant women/children

Be scientific, evidence-based, and thorough. Reference specific studies or regulatory data when possible.`;

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { ingredients, productType } = await req.json();
    
    if (!ingredients || typeof ingredients !== "string") {
      return new Response(
        JSON.stringify({ error: "Ingredients string is required" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const LOVABLE_API_KEY = Deno.env.get("LOVABLE_API_KEY");
    if (!LOVABLE_API_KEY) {
      console.error("LOVABLE_API_KEY is not configured");
      throw new Error("AI service is not configured");
    }

    const userPrompt = `Analyze the following ingredient list from a ${productType || "cosmetic/personal care"} product using the TOBIN framework:

INGREDIENTS:
${ingredients}

Provide:
1. Complete TOBIN scores (T, O, B, I, N) with the summary table
2. Detailed analysis of each concerning ingredient
3. List of safe ingredients
4. Comprehensive recommendations
5. Any special warnings for vulnerable populations

Be thorough and scientific. Calculate each TOBIN category score carefully based on the ingredients present.`;

    console.log("Analyzing ingredients with TOBIN framework, length:", ingredients.length);

    const response = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${LOVABLE_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-3-flash-preview",
        messages: [
          { role: "system", content: SYSTEM_PROMPT },
          { role: "user", content: userPrompt },
        ],
        stream: true,
      }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error("AI gateway error:", response.status, errorText);
      
      if (response.status === 429) {
        return new Response(
          JSON.stringify({ error: "Rate limit exceeded. Please wait a moment and try again." }),
          { status: 429, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      
      if (response.status === 402) {
        return new Response(
          JSON.stringify({ error: "AI service credits exhausted. Please try again later." }),
          { status: 402, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      
      return new Response(
        JSON.stringify({ error: "Failed to analyze ingredients" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    console.log("Streaming TOBIN ingredient analysis response");

    return new Response(response.body, {
      headers: {
        ...corsHeaders,
        "Content-Type": "text/event-stream",
        "Cache-Control": "no-cache",
        "Connection": "keep-alive",
      },
    });
  } catch (error) {
    console.error("Ingredient analysis error:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
