import "https://deno.land/x/xhr@0.1.0/mod.ts";
import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { recipeContent, servings } = await req.json();

    if (!recipeContent) {
      throw new Error('Recipe content is required');
    }

    const systemPrompt = `You are a nutritional analysis expert. Analyze recipes and estimate their nutritional information.

Given a recipe, estimate the nutritional information per serving. Be practical and use common nutritional databases as reference.

Return ONLY valid JSON in this exact format:
{
  "calories": 450,
  "protein": 35,
  "carbs": 25,
  "fat": 24,
  "fiber": 6,
  "sugar": 4,
  "sodium": 580,
  "highlights": [
    "High in omega-3 fatty acids",
    "Low glycemic impact",
    "Rich in antioxidants"
  ],
  "warnings": []
}

Notes:
- All values are per serving
- calories in kcal
- protein, carbs, fat, fiber, sugar in grams
- sodium in mg
- highlights: 2-4 key nutritional benefits
- warnings: any concerns (empty array if none)
- Be realistic with estimates based on typical ingredients`;

    const response = await fetch("https://api.dev.cloud.lovable.dev/chat", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "x-lovable-functions-secret": Deno.env.get("LOVABLE_API_KEY") ?? "",
      },
      body: JSON.stringify({
        model: "google/gemini-2.5-flash",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: `Analyze this recipe for ${servings || 2} servings and provide nutritional information per serving:\n\n${recipeContent}` }
        ],
      }),
    });

    if (!response.ok) {
      throw new Error(`API request failed: ${response.status}`);
    }

    const data = await response.json();
    const content = data.choices?.[0]?.message?.content || '';
    
    // Parse JSON from the response
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      throw new Error('Failed to parse nutritional info from response');
    }
    
    const nutritionalInfo = JSON.parse(jsonMatch[0]);

    return new Response(JSON.stringify(nutritionalInfo), {
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });
  } catch (error: unknown) {
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    console.error('Error analyzing nutrition:', error);
    return new Response(JSON.stringify({ error: errorMessage }), {
      status: 500,
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });
  }
});
