import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

// Known phytoestrogen-containing ingredients for quick matching
const PHYTOESTROGEN_KEYWORDS = {
  strong: [
    'soy', 'soya', 'soja', 'tofu', 'tempeh', 'edamame', 'miso', 'natto',
    'soy protein', 'soy lecithin', 'soya lecithin', 'lecithine de soja', 'lécithine de soja',
    'soybean', 'soy flour', 'soy milk', 'soya milk', 'lait de soja',
    'flaxseed', 'flax seed', 'linseed', 'flax', 'lin', 'graines de lin',
    'red clover', 'kudzu', 'pueraria'
  ],
  moderate: [
    'chickpea', 'garbanzo', 'hummus', 'pois chiche', 'lentil', 'lentille',
    'sesame', 'sésame', 'tahini', 'tahina',
    'oat', 'avoine', 'barley', 'orge', 'rye', 'seigle', 'wheat bran', 'son de blé', 'bran',
    'licorice', 'réglisse', 'hops', 'houblon', 'beer', 'bière',
    'fennel', 'fenouil', 'anise', 'anis',
    'alfalfa', 'luzerne', 'clover', 'trèfle', 'fenugreek', 'fenugrec'
  ],
  weak: [
    'sunflower', 'tournesol', 'pumpkin seed', 'graines de courge',
    'walnut', 'noix', 'pecan', 'pécan', 'pistachio', 'pistache',
    'olive oil', 'huile d\'olive', 'broccoli', 'brocoli', 'cabbage', 'chou',
    'kale', 'brussels', 'choux de bruxelles',
    'berries', 'baies', 'strawberry', 'fraise', 'raspberry', 'framboise',
    'blueberry', 'myrtille', 'cranberry', 'canneberge',
    'apple', 'pomme', 'pear', 'poire', 'cherry', 'cerise', 'plum', 'prune',
    'coffee', 'café', 'tea', 'thé', 'wine', 'vin', 'grape', 'raisin'
  ]
};

interface PhytoAnalysis {
  detected: boolean;
  overallRisk: 'none' | 'low' | 'moderate' | 'high';
  compounds: {
    name: string;
    source: string;
    potency: 'weak' | 'moderate' | 'strong';
  }[];
  summary: string;
  recommendation: string;
}

function quickAnalyze(ingredients: string): PhytoAnalysis {
  const ingredientsLower = ingredients.toLowerCase();
  const detected: PhytoAnalysis['compounds'] = [];

  // Check strong phytoestrogens
  for (const keyword of PHYTOESTROGEN_KEYWORDS.strong) {
    if (ingredientsLower.includes(keyword)) {
      detected.push({
        name: keyword.includes('soy') ? 'Isoflavones (Genistein, Daidzein)' : 
              keyword.includes('flax') ? 'Lignans (Secoisolariciresinol)' : 
              'Isoflavones',
        source: keyword.charAt(0).toUpperCase() + keyword.slice(1),
        potency: 'strong'
      });
    }
  }

  // Check moderate phytoestrogens
  for (const keyword of PHYTOESTROGEN_KEYWORDS.moderate) {
    if (ingredientsLower.includes(keyword)) {
      detected.push({
        name: keyword.includes('sesame') ? 'Lignans (Sesamin)' :
              keyword.includes('oat') || keyword.includes('barley') ? 'Lignans' :
              keyword.includes('hops') || keyword.includes('beer') ? 'Prenylnaringenin' :
              'Isoflavones/Lignans',
        source: keyword.charAt(0).toUpperCase() + keyword.slice(1),
        potency: 'moderate'
      });
    }
  }

  // Check weak phytoestrogens
  for (const keyword of PHYTOESTROGEN_KEYWORDS.weak) {
    if (ingredientsLower.includes(keyword)) {
      detected.push({
        name: 'Trace Lignans/Flavonoids',
        source: keyword.charAt(0).toUpperCase() + keyword.slice(1),
        potency: 'weak'
      });
    }
  }

  // Deduplicate by source
  const uniqueCompounds = detected.filter((item, index, self) =>
    index === self.findIndex((t) => t.source.toLowerCase() === item.source.toLowerCase())
  );

  // Determine overall risk
  const hasStrong = uniqueCompounds.some(c => c.potency === 'strong');
  const hasModerate = uniqueCompounds.some(c => c.potency === 'moderate');
  const overallRisk: PhytoAnalysis['overallRisk'] = 
    hasStrong ? 'high' : 
    hasModerate ? 'moderate' : 
    uniqueCompounds.length > 0 ? 'low' : 'none';

  return {
    detected: uniqueCompounds.length > 0,
    overallRisk,
    compounds: uniqueCompounds.slice(0, 5), // Limit to top 5
    summary: uniqueCompounds.length === 0 
      ? 'No significant phytoestrogen sources detected in ingredients.'
      : `Found ${uniqueCompounds.length} potential phytoestrogen source(s).`,
    recommendation: overallRisk === 'high' 
      ? 'This product contains significant phytoestrogen sources. Consider limiting intake if avoiding estrogenic compounds.'
      : overallRisk === 'moderate'
      ? 'Contains moderate phytoestrogen sources. Monitor consumption if sensitive.'
      : overallRisk === 'low'
      ? 'Contains trace phytoestrogens. Generally low concern for most people.'
      : 'No phytoestrogen concerns identified.'
  };
}

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const { ingredients, productName, useAI = false } = await req.json();

    if (!ingredients) {
      return new Response(
        JSON.stringify({ error: 'Ingredients list is required' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log(`Analyzing phytoestrogens for: ${productName || 'Unknown product'}`);

    // Quick keyword-based analysis (fast, no AI needed)
    const quickResult = quickAnalyze(ingredients);

    // If AI analysis requested and we found phytoestrogens, enhance with AI
    if (useAI && quickResult.detected) {
      const LOVABLE_API_KEY = Deno.env.get('LOVABLE_API_KEY');
      
      if (LOVABLE_API_KEY) {
        try {
          const aiResponse = await fetch('https://ai.gateway.lovable.dev/v1/chat/completions', {
            method: 'POST',
            headers: {
              'Authorization': `Bearer ${LOVABLE_API_KEY}`,
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({
              model: 'google/gemini-2.5-flash',
              messages: [
                {
                  role: 'system',
                  content: `You are an expert in phytoestrogens and endocrine-affecting compounds in food. Analyze ingredients for phytoestrogen content. Be concise and factual. Focus on: isoflavones (soy), lignans (flax, sesame), coumestans, and other estrogenic compounds.`
                },
                {
                  role: 'user',
                  content: `Analyze these ingredients for phytoestrogen content: "${ingredients}". 
                  
Product: ${productName || 'Unknown'}

Provide a brief (2-3 sentence) health-focused summary about the phytoestrogen exposure from this product.`
                }
              ],
              max_tokens: 200,
            }),
          });

          if (aiResponse.ok) {
            const aiData = await aiResponse.json();
            const aiSummary = aiData.choices?.[0]?.message?.content;
            if (aiSummary) {
              quickResult.summary = aiSummary;
            }
          }
        } catch (aiError) {
          console.error('AI enhancement failed, using quick analysis:', aiError);
        }
      }
    }

    console.log(`Analysis complete: ${quickResult.overallRisk} risk, ${quickResult.compounds.length} compounds found`);

    return new Response(
      JSON.stringify(quickResult),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error) {
    console.error('Error analyzing phytoestrogens:', error);
    return new Response(
      JSON.stringify({ error: 'Failed to analyze ingredients' }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
