import { createClient } from 'https://esm.sh/@supabase/supabase-js@2';

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

interface MealPlanWithReminder {
  id: string;
  user_id: string;
  recipe_id: string;
  planned_date: string;
  meal_slot: string;
  reminder_time: string;
  recipe: {
    title: string;
  } | null;
}

Deno.serve(async (req) => {
  // Handle CORS preflight requests
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get('SUPABASE_URL')!;
    const supabaseServiceKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!;
    const supabase = createClient(supabaseUrl, supabaseServiceKey);

    // Parse request body to get user_id (for client-triggered checks)
    let userId: string | null = null;
    try {
      const body = await req.json();
      userId = body.user_id;
    } catch {
      // No body provided, will check all users
    }

    const now = new Date();
    const today = now.toISOString().split('T')[0];
    const tomorrow = new Date(now.getTime() + 24 * 60 * 60 * 1000).toISOString().split('T')[0];
    const currentHour = now.getHours();

    console.log(`Checking reminders at ${now.toISOString()}, hour: ${currentHour}`);
    console.log(`Today: ${today}, Tomorrow: ${tomorrow}`);

    // Build query for meal plans with reminders
    let query = supabase
      .from('meal_plans')
      .select(`
        id,
        user_id,
        recipe_id,
        planned_date,
        meal_slot,
        reminder_time,
        recipe:saved_recipes(title)
      `)
      .not('reminder_time', 'is', null)
      .in('planned_date', [today, tomorrow]);

    if (userId) {
      query = query.eq('user_id', userId);
    }

    const { data: mealPlans, error: fetchError } = await query;

    if (fetchError) {
      console.error('Error fetching meal plans:', fetchError);
      throw fetchError;
    }

    console.log(`Found ${mealPlans?.length || 0} meal plans with reminders`);

    const notifications: {
      user_id: string;
      type: string;
      title: string;
      message: string;
      link: string;
      metadata: Record<string, unknown>;
    }[] = [];

    // Process each meal plan
    const plans = mealPlans || [];
    for (let i = 0; i < plans.length; i++) {
      const plan = plans[i];
      const recipeData = Array.isArray(plan.recipe) ? plan.recipe[0] : plan.recipe;
      const recipeTitle = recipeData?.title || 'your meal';
      const mealDate = plan.planned_date;
      const reminderType = plan.reminder_time;

      // Determine if we should send this reminder based on current time
      let shouldSend = false;
      let reminderMessage = '';

      if (mealDate === tomorrow) {
        // Day before reminders
        if (reminderType === 'morning_before' && currentHour >= 8 && currentHour < 10) {
          shouldSend = true;
          reminderMessage = `Tomorrow's ${plan.meal_slot}: ${recipeTitle}. Time to prep ingredients!`;
        } else if (reminderType === 'evening_before' && currentHour >= 17 && currentHour < 19) {
          shouldSend = true;
          reminderMessage = `Tomorrow's ${plan.meal_slot}: ${recipeTitle}. Check if you have all ingredients!`;
        }
      } else if (mealDate === today) {
        // Same day reminder
        if (reminderType === 'morning_of' && currentHour >= 6 && currentHour < 8) {
          shouldSend = true;
          reminderMessage = `Today's ${plan.meal_slot}: ${recipeTitle}. Time to start cooking!`;
        }
      }

      if (shouldSend) {
        // Check if we already sent this notification (prevent duplicates)
        const { data: existingNotif } = await supabase
          .from('notifications')
          .select('id')
          .eq('user_id', plan.user_id)
          .eq('type', 'meal_reminder')
          .gte('created_at', new Date(now.getTime() - 6 * 60 * 60 * 1000).toISOString())
          .limit(1);

        if (!existingNotif || existingNotif.length === 0) {
          notifications.push({
            user_id: plan.user_id,
            type: 'meal_reminder',
            title: '🍳 Meal Prep Reminder',
            message: reminderMessage,
            link: '/recipes',
            metadata: {
              meal_plan_id: plan.id,
              recipe_id: plan.recipe_id,
              planned_date: plan.planned_date,
              meal_slot: plan.meal_slot,
            },
          });
        }
      }
    }

    console.log(`Creating ${notifications.length} notifications`);

    // Insert all notifications
    if (notifications.length > 0) {
      const { error: insertError } = await supabase
        .from('notifications')
        .insert(notifications);

      if (insertError) {
        console.error('Error creating notifications:', insertError);
        throw insertError;
      }
    }

    return new Response(
      JSON.stringify({
        success: true,
        checked: mealPlans?.length || 0,
        notificationsSent: notifications.length,
      }),
      {
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  } catch (error) {
    console.error('Error in check-meal-reminders:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    return new Response(
      JSON.stringify({ error: errorMessage }),
      {
        status: 500,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  }
});