import "https://deno.land/x/xhr@0.1.0/mod.ts";
import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { recipes } = await req.json();

    if (!recipes || !Array.isArray(recipes) || recipes.length === 0) {
      throw new Error('At least one recipe is required');
    }

    const combinedRecipes = recipes.map((r: { title: string; content: string }) => 
      `=== ${r.title} ===\n${r.content}`
    ).join('\n\n');

    const systemPrompt = `You are a helpful assistant that extracts ingredients from recipes and creates organized shopping lists.

Your task:
1. Parse the provided recipes and extract ALL ingredients
2. Combine duplicate ingredients and sum their quantities
3. Organize ingredients by category (Produce, Proteins, Dairy, Pantry, Spices & Seasonings, Oils & Fats, Other)
4. Use consistent units (convert when sensible - e.g., 16 tbsp butter = 1 cup)

Return ONLY valid JSON in this exact format:
{
  "categories": [
    {
      "name": "Produce",
      "items": [
        { "name": "Avocados", "quantity": "4", "unit": "medium" },
        { "name": "Spinach", "quantity": "8", "unit": "cups" }
      ]
    }
  ],
  "summary": {
    "totalItems": 15,
    "recipesProcessed": 2
  }
}

Important:
- Be thorough - extract EVERY ingredient mentioned
- Use practical quantities for shopping (e.g., "2 lbs" not "907g")
- Include common names people would look for in a store
- Group similar items (all types of peppers under "Bell peppers, assorted" if appropriate)`;

    const response = await fetch("https://api.dev.cloud.lovable.dev/chat", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "x-lovable-functions-secret": Deno.env.get("LOVABLE_API_KEY") ?? "",
      },
      body: JSON.stringify({
        model: "google/gemini-2.5-flash",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: `Extract ingredients from these recipes:\n\n${combinedRecipes}` }
        ],
      }),
    });

    if (!response.ok) {
      throw new Error(`API request failed: ${response.status}`);
    }

    const data = await response.json();
    const content = data.choices?.[0]?.message?.content || '';
    
    // Parse JSON from the response
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      throw new Error('Failed to parse ingredients from response');
    }
    
    const ingredients = JSON.parse(jsonMatch[0]);

    return new Response(JSON.stringify(ingredients), {
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });
  } catch (error: unknown) {
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    console.error('Error extracting ingredients:', error);
    return new Response(JSON.stringify({ error: errorMessage }), {
      status: 500,
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });
  }
});
