import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

// Foods that are SAFE (no phytoestrogens or have anti-estrogenic properties)
const PHYTOESTROGEN_FREE_FOODS = `
PHYTOESTROGEN-FREE THERAPEUTIC FOODS (Use freely):
- Wild salmon - Rich in omega-3s, no estrogenic compounds
- Sardines - High protein, anti-inflammatory
- Pasture-raised eggs - Complete protein, cholesterol supports hormone balance
- Grass-fed beef liver - Nutrient dense, supports detox pathways
- Grass-fed beef - High quality protein, zinc for hormone balance
- Lamb - Excellent protein source
- Chicken breast - Lean protein
- Turkey - Low fat protein
- Shrimp - Iodine rich, thyroid support
- Cod - Lean white fish
- Tuna - Protein and selenium
- Avocado - Healthy fats, no estrogenic activity
- Extra virgin olive oil - Anti-inflammatory, hormone neutral
- Coconut oil - MCTs, no estrogenic effects
- Grass-fed butter/ghee - Fat-soluble vitamins
- Broccoli - Contains Indole-3-carbinol (ANTI-estrogenic)
- Cauliflower - Cruciferous, supports estrogen metabolism
- Brussels sprouts - DIM precursors, anti-estrogenic
- Cabbage - Cruciferous detox support
- Kale - Anti-estrogenic compounds
- Spinach - Iron rich, no significant phytoestrogens
- Asparagus - Prebiotic fiber
- Zucchini - Low calorie, hormone neutral
- Bell peppers - Vitamin C rich
- Mushrooms - Immune support, aromatase inhibiting
- Garlic - Antimicrobial, hormone neutral
- Ginger - Anti-inflammatory
- Turmeric - Curcumin benefits
- Onions - Quercetin, no estrogenic activity
- Celery - Low calorie, mineral rich
- Cucumber - Hydrating, hormone neutral
- Bone broth - Gut healing
- Macadamia nuts - Lowest phytoestrogen content of nuts
- Pumpkin seeds - Zinc for testosterone support
- Lemons/Limes - Alkalizing
- Berries (small amounts) - Low sugar, antioxidants
- Green tea - EGCG may have anti-estrogenic effects
- Apple cider vinegar - Digestive support
`;

const FOODS_TO_STRICTLY_AVOID = `
FOODS TO STRICTLY AVOID (High phytoestrogen content):
❌ SOY PRODUCTS (highest estrogenic):
  - Tofu, tempeh, edamame, soy milk, soy protein, miso, soy sauce
  - Contains genistein and daidzein (potent isoflavones)

❌ LEGUMES (moderate to high estrogenic):
  - Chickpeas, lentils, black beans, kidney beans, split peas
  - Hummus and all bean-based dishes

❌ FLAXSEEDS (very high lignans):
  - Flax oil, ground flax, flax crackers
  - Highest lignan content of any food

❌ SESAME (high lignans):
  - Tahini, sesame oil, sesame seeds

❌ CERTAIN GRAINS:
  - Oats (contain lignans and avenanthramides)
  - Wheat bran, wheat germ
  - Barley

❌ OTHER HIGH-ESTROGEN FOODS:
  - Dried fruits (concentrated)
  - Alfalfa sprouts
  - Sunflower seeds
  - Hops (beer)
  - Licorice root
  - Red clover
  - Fennel
`;

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { mealType, servings, preferences } = await req.json();
    
    const LOVABLE_API_KEY = Deno.env.get("LOVABLE_API_KEY");
    if (!LOVABLE_API_KEY) {
      throw new Error("LOVABLE_API_KEY is not configured");
    }

    const systemPrompt = `You are a hormone-health chef specializing in creating delicious recipes that are 100% FREE of phytoestrogens and xenoestrogens. Your goal is to help users minimize estrogenic compound exposure while enjoying satisfying, nutrient-dense meals.

CRITICAL RULES - NEVER USE THESE INGREDIENTS:
1. NO soy products whatsoever (tofu, tempeh, soy sauce, edamame, soy milk)
2. NO legumes/beans (chickpeas, lentils, black beans, kidney beans, hummus)
3. NO flaxseeds or flax products
4. NO sesame seeds or tahini
5. NO oats or oat products
6. NO beer or hops-containing products

FOCUS ON THESE APPROVED INGREDIENTS ONLY:
${PHYTOESTROGEN_FREE_FOODS}

${FOODS_TO_STRICTLY_AVOID}

RECIPE GUIDELINES:
- Use ONLY ingredients from the approved list
- Emphasize cruciferous vegetables (broccoli, cauliflower, Brussels sprouts, cabbage) for their ANTI-estrogenic properties
- Include high-quality proteins (grass-fed meats, wild fish, pasture-raised eggs)
- Use healthy fats (olive oil, coconut oil, avocado, grass-fed butter)
- Season with garlic, ginger, turmeric, herbs and spices
- Create flavorful meals that don't feel restrictive
- Include the anti-estrogenic benefits of each meal`;

    const userPrompt = `Create a complete ${mealType || 'daily'} PHYTOESTROGEN-FREE meal plan with ${servings || 2} servings per recipe.
${preferences ? `Preferences: ${preferences}` : ''}

IMPORTANT: Every single ingredient must be from the approved phytoestrogen-free list. Double-check that NO legumes, soy, flax, sesame, or oats are included.

For each meal, provide:
1. Recipe name (creative and appetizing)
2. "🛡️ Hormone-Safe" badge confirming zero phytoestrogens
3. Brief description highlighting anti-estrogenic benefits
4. Prep time and cook time
5. Complete ingredient list with quantities (ONLY approved ingredients)
6. Step-by-step instructions
7. "Anti-Estrogenic Highlights" - which ingredients actively support hormone balance
8. Chef's tips for best results

Include: breakfast, lunch, dinner, and 1-2 hormone-balancing snacks.

End with a summary of the total anti-estrogenic benefits of this meal plan.`;

    const response = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${LOVABLE_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-3-flash-preview",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: userPrompt },
        ],
        stream: true,
      }),
    });

    if (!response.ok) {
      if (response.status === 429) {
        return new Response(JSON.stringify({ error: "Rate limit exceeded. Please try again in a moment." }), {
          status: 429,
          headers: { ...corsHeaders, "Content-Type": "application/json" },
        });
      }
      if (response.status === 402) {
        return new Response(JSON.stringify({ error: "AI credits exhausted. Please add credits to continue." }), {
          status: 402,
          headers: { ...corsHeaders, "Content-Type": "application/json" },
        });
      }
      const errorText = await response.text();
      console.error("AI gateway error:", response.status, errorText);
      return new Response(JSON.stringify({ error: "Failed to generate recipes" }), {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    return new Response(response.body, {
      headers: { ...corsHeaders, "Content-Type": "text/event-stream" },
    });
  } catch (error) {
    console.error("Phytoestrogen-free recipe generation error:", error);
    return new Response(JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }), {
      status: 500,
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  }
});
