import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

const THERAPEUTIC_FOODS = `
THERAPEUTIC FOODS (LI < 0.5, GI* < 0.6 - Promotes fat loss):
- Wild salmon (LI: 0.12, GI*: 0.10) - Rich in omega-3s
- Sardines (LI: 0.15, GI*: 0.05) - High protein, low glycemic
- Pasture-raised eggs (LI: 0.18, GI*: 0.02) - Complete protein
- Grass-fed beef liver (LI: 0.20, GI*: 0.03) - Nutrient dense
- Avocado (LI: 0.22, GI*: 0.15) - Healthy fats
- Extra virgin olive oil (LI: 0.08, GI*: 0.01) - Anti-inflammatory
- Coconut oil (LI: 0.10, GI*: 0.01) - MCTs for energy
- Broccoli (LI: 0.05, GI*: 0.12) - Detoxifying
- Spinach (LI: 0.03, GI*: 0.08) - Iron and minerals
- Kale (LI: 0.04, GI*: 0.10) - Antioxidants
- Cauliflower (LI: 0.05, GI*: 0.11) - Versatile low-carb
- Asparagus (LI: 0.04, GI*: 0.09) - Prebiotic fiber
- Brussels sprouts (LI: 0.06, GI*: 0.14) - Sulfur compounds
- Zucchini (LI: 0.03, GI*: 0.10) - Low calorie
- Bell peppers (LI: 0.05, GI*: 0.13) - Vitamin C rich
- Mushrooms (LI: 0.02, GI*: 0.05) - Immune support
- Garlic (LI: 0.04, GI*: 0.12) - Antimicrobial
- Ginger (LI: 0.03, GI*: 0.08) - Anti-inflammatory
- Turmeric (LI: 0.02, GI*: 0.06) - Curcumin benefits
- Bone broth (LI: 0.08, GI*: 0.02) - Gut healing
- Grass-fed butter (LI: 0.15, GI*: 0.01) - Fat-soluble vitamins
- Almonds (LI: 0.25, GI*: 0.18) - Healthy snack
- Walnuts (LI: 0.22, GI*: 0.15) - Brain food
- Macadamia nuts (LI: 0.20, GI*: 0.12) - Lowest carb nut
- Chia seeds (LI: 0.18, GI*: 0.10) - Omega-3 and fiber
- Flax seeds (LI: 0.15, GI*: 0.08) - Lignans
- Hemp seeds (LI: 0.20, GI*: 0.12) - Complete protein
- Lemon (LI: 0.02, GI*: 0.08) - Alkalizing
- Lime (LI: 0.02, GI*: 0.07) - Vitamin C
- Berries (LI: 0.28, GI*: 0.32) - Low sugar fruits
- Green tea (LI: 0.01, GI*: 0.01) - EGCG antioxidants
`;

const SAFE_FOODS = `
METABOLICALLY SAFE FOODS (Foundation foods):
- Chicken breast (LI: 0.35, GI*: 0.08)
- Turkey (LI: 0.32, GI*: 0.06)
- Grass-fed beef (LI: 0.40, GI*: 0.05)
- Lamb (LI: 0.38, GI*: 0.04)
- Shrimp (LI: 0.18, GI*: 0.05)
- Cod (LI: 0.15, GI*: 0.03)
- Tuna (LI: 0.20, GI*: 0.04)
- Greek yogurt (full-fat) (LI: 0.45, GI*: 0.28)
- Cheese (aged) (LI: 0.42, GI*: 0.08)
- Cottage cheese (LI: 0.40, GI*: 0.15)
- Sweet potato (LI: 0.55, GI*: 0.48)
- Quinoa (LI: 0.52, GI*: 0.45)
- Lentils (LI: 0.48, GI*: 0.42)
- Black beans (LI: 0.45, GI*: 0.38)
- Chickpeas (LI: 0.50, GI*: 0.40)
`;

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { mealType, servings, dietaryRestrictions, preferences } = await req.json();
    
    const LOVABLE_API_KEY = Deno.env.get("LOVABLE_API_KEY");
    if (!LOVABLE_API_KEY) {
      throw new Error("LOVABLE_API_KEY is not configured");
    }

    const systemPrompt = `You are a metabolic health chef specializing in creating delicious recipes using low Lipogenic Index (LI) and low Glycemic Index (GI*) foods. Your recipes promote fat loss and metabolic health.

IMPORTANT GUIDELINES:
- Focus on THERAPEUTIC and SAFE foods with LI < 0.5 and GI* < 0.6
- Avoid high-glycemic ingredients like refined sugars, white flour, and processed foods
- Create flavorful, satisfying meals that don't feel restrictive
- Include cooking tips and nutritional benefits
- Make recipes practical and achievable

${THERAPEUTIC_FOODS}

${SAFE_FOODS}

FOODS TO AVOID (High LI/GI*):
- White bread, pasta, rice
- Sugar and sweeteners (except stevia/monk fruit)
- Processed foods
- Seed oils (canola, soybean, corn)
- High-fructose fruits
- Processed meats with additives`;

    const userPrompt = `Create a complete ${mealType || 'daily'} meal plan with ${servings || 2} servings per recipe.
${dietaryRestrictions ? `Dietary restrictions: ${dietaryRestrictions}` : ''}
${preferences ? `Preferences: ${preferences}` : ''}

For each meal, provide:
1. Recipe name (creative and appetizing)
2. Brief description highlighting metabolic benefits
3. Prep time and cook time
4. Complete ingredient list with quantities
5. Step-by-step instructions
6. Nutritional highlights (key LI/GI* friendly ingredients)
7. Chef's tips for best results

Make the recipes delicious, practical, and focused on metabolic health. Include breakfast, lunch, dinner, and 1-2 healthy snacks.`;

    const response = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${LOVABLE_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-3-flash-preview",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: userPrompt },
        ],
        stream: true,
      }),
    });

    if (!response.ok) {
      if (response.status === 429) {
        return new Response(JSON.stringify({ error: "Rate limit exceeded. Please try again in a moment." }), {
          status: 429,
          headers: { ...corsHeaders, "Content-Type": "application/json" },
        });
      }
      if (response.status === 402) {
        return new Response(JSON.stringify({ error: "AI credits exhausted. Please add credits to continue." }), {
          status: 402,
          headers: { ...corsHeaders, "Content-Type": "application/json" },
        });
      }
      const errorText = await response.text();
      console.error("AI gateway error:", response.status, errorText);
      return new Response(JSON.stringify({ error: "Failed to generate recipes" }), {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    return new Response(response.body, {
      headers: { ...corsHeaders, "Content-Type": "text/event-stream" },
    });
  } catch (error) {
    console.error("Recipe generation error:", error);
    return new Response(JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }), {
      status: 500,
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  }
});
