import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

const SYSTEM_PROMPT = `You are a knowledgeable, evidence-based health and nutrition advisor specializing in metabolic health, weight management, and chronic disease prevention. 

Your expertise includes:
- The Lipogenic Index (LI) and Glycogenic Index (GI*) framework for food analysis
- Insulin biology and its role in weight gain
- AMPK activation and mTOR signaling
- Mitochondrial health and metabolic efficiency
- De novo lipogenesis and fat storage mechanisms
- Practical dietary strategies for weight loss and metabolic health

CRITICAL GUIDELINES:
1. ALWAYS provide evidence-based information. If uncertain, clearly state "I'm not certain about this, but..."
2. NEVER diagnose medical conditions. Always recommend consulting healthcare professionals for medical advice.
3. Focus on education about metabolic mechanisms and food choices.
4. When discussing specific foods, reference their LI and GI* properties when relevant.
5. Provide actionable, practical recommendations.
6. Be clear about what is established science vs. emerging research.
7. Avoid extreme claims - use measured, scientific language.

KEY PRINCIPLES TO EMPHASIZE:
- Weight gain is driven by insulin-driven fat storage, not just calories
- High LI foods (>1.0) promote fat storage; low LI foods (<0.5) are metabolically safe
- High GI* foods rapidly fill glycogen and spike insulin
- The combination of carbs + fat is particularly lipogenic
- Protein + Fiber + Healthy Fat = optimal meal structure
- AMPK activators (olive oil, green tea, vinegar) support fat oxidation
- Avoid "naked carbohydrates" - always pair with protein and fiber

When asked about specific health conditions, provide general educational information but always recommend professional medical consultation. For nutrition questions, be specific and actionable.

Format responses clearly with:
- Bullet points for lists
- Bold text for key terms (using **bold**)
- Organized sections for complex topics`;

serve(async (req) => {
  // Handle CORS preflight
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { messages } = await req.json();
    
    if (!messages || !Array.isArray(messages)) {
      return new Response(
        JSON.stringify({ error: "Messages array is required" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const LOVABLE_API_KEY = Deno.env.get("LOVABLE_API_KEY");
    if (!LOVABLE_API_KEY) {
      console.error("LOVABLE_API_KEY is not configured");
      throw new Error("AI service is not configured");
    }

    console.log("Sending request to Lovable AI with", messages.length, "messages");

    const response = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${LOVABLE_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-3-flash-preview",
        messages: [
          { role: "system", content: SYSTEM_PROMPT },
          ...messages,
        ],
        stream: true,
      }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error("AI gateway error:", response.status, errorText);
      
      if (response.status === 429) {
        return new Response(
          JSON.stringify({ error: "Rate limit exceeded. Please wait a moment and try again." }),
          { status: 429, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      
      if (response.status === 402) {
        return new Response(
          JSON.stringify({ error: "AI service credits exhausted. Please try again later." }),
          { status: 402, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      
      return new Response(
        JSON.stringify({ error: "Failed to get AI response" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    console.log("Streaming response from Lovable AI");

    return new Response(response.body, {
      headers: {
        ...corsHeaders,
        "Content-Type": "text/event-stream",
        "Cache-Control": "no-cache",
        "Connection": "keep-alive",
      },
    });
  } catch (error) {
    console.error("Health chat error:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
