import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from 'https://esm.sh/@supabase/supabase-js@2';

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

interface ProductResult {
  name: string;
  brand: string | null;
  categories: string | null;
  imageUrl: string | null;
  ingredients: string | null;
  nutrition?: {
    calories: number | null;
    fat: number | null;
    carbs: number | null;
    protein: number | null;
    fiber: number | null;
    sugar: number | null;
    salt: number | null;
  } | null;
  nutriscoreGrade?: string | null;
  novaGroup?: number | null;
}

interface AIAnalysis {
  summary: string;
  safetyRating: 'Safe' | 'Moderate Concern' | 'High Concern';
  keyIngredients: string[];
  concerns: string[];
  benefits: string[];
  suitableFor: string[];
  avoidIf: string[];
  alternatives: string[];
  sources?: string[];
}

interface LookupResult {
  found: boolean;
  product?: ProductResult;
  source?: string;
  productType?: 'food' | 'cosmetic';
  aiGenerated?: boolean;
  aiAnalysis?: AIAnalysis | null;
}

// Check community-contributed products database first
async function lookupCommunityProducts(barcode: string): Promise<LookupResult> {
  try {
    const supabaseUrl = Deno.env.get('SUPABASE_URL')!;
    const supabaseKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    const { data, error } = await supabase
      .from('community_products')
      .select('*')
      .eq('barcode', barcode)
      .single();

    if (error || !data) {
      return { found: false };
    }

    console.log(`Found in community database: ${data.product_name}`);
    return {
      found: true,
      source: 'Community Database',
      productType: data.product_type as 'food' | 'cosmetic',
      product: {
        name: data.product_name,
        brand: data.brand,
        categories: data.categories,
        imageUrl: data.image_url,
        ingredients: data.ingredients,
      },
    };
  } catch (error) {
    console.error('Community database lookup error:', error);
    return { found: false };
  }
}

// Open Food Facts - comprehensive food database
async function lookupOpenFoodFacts(barcode: string): Promise<LookupResult> {
  try {
    const response = await fetch(
      `https://world.openfoodfacts.org/api/v2/product/${barcode}.json`,
      {
        method: 'GET',
        headers: { 'User-Agent': 'PhytoestrogenTracker/1.0' },
      }
    );

    if (!response.ok) {
      console.log(`Open Food Facts: ${response.status} for ${barcode}`);
      return { found: false };
    }

    const data = await response.json();
    if (data.status === 1 && data.product) {
      const p = data.product;
      return {
        found: true,
        source: 'Open Food Facts',
        productType: 'food',
        product: {
          name: p.product_name || p.brands || 'Unknown Product',
          brand: p.brands || null,
          categories: p.categories || null,
          imageUrl: p.image_url || null,
          ingredients: p.ingredients_text || null,
          nutrition: p.nutriments ? {
            calories: p.nutriments.energy_kcal_100g || null,
            fat: p.nutriments.fat_100g || null,
            carbs: p.nutriments.carbohydrates_100g || null,
            protein: p.nutriments.proteins_100g || null,
            fiber: p.nutriments.fiber_100g || null,
            sugar: p.nutriments.sugars_100g || null,
            salt: p.nutriments.salt_100g || null,
          } : null,
          nutriscoreGrade: p.nutriscore_grade || null,
          novaGroup: p.nova_group || null,
        },
      };
    }
    return { found: false };
  } catch (error) {
    console.error('Open Food Facts error:', error);
    return { found: false };
  }
}

// Open Beauty Facts - cosmetics database
async function lookupOpenBeautyFacts(barcode: string): Promise<LookupResult> {
  try {
    const response = await fetch(
      `https://world.openbeautyfacts.org/api/v2/product/${barcode}.json`,
      {
        method: 'GET',
        headers: { 'User-Agent': 'PhytoestrogenTracker/1.0' },
      }
    );

    if (!response.ok) {
      console.log(`Open Beauty Facts: ${response.status} for ${barcode}`);
      return { found: false };
    }

    const data = await response.json();
    if (data.status === 1 && data.product) {
      const p = data.product;
      return {
        found: true,
        source: 'Open Beauty Facts',
        productType: 'cosmetic',
        product: {
          name: p.product_name || p.brands || 'Unknown Product',
          brand: p.brands || null,
          categories: p.categories || null,
          imageUrl: p.image_url || null,
          ingredients: p.ingredients_text || null,
        },
      };
    }
    return { found: false };
  } catch (error) {
    console.error('Open Beauty Facts error:', error);
    return { found: false };
  }
}

// Open Products Facts - general products database
async function lookupOpenProductsFacts(barcode: string): Promise<LookupResult> {
  try {
    const response = await fetch(
      `https://world.openproductsfacts.org/api/v2/product/${barcode}.json`,
      {
        method: 'GET',
        headers: { 'User-Agent': 'PhytoestrogenTracker/1.0' },
      }
    );

    if (!response.ok) return { found: false };

    const data = await response.json();
    if (data.status === 1 && data.product) {
      const p = data.product;
      const categories = (p.categories || '').toLowerCase();
      const isCosmetic = categories.includes('cosmetic') || 
                         categories.includes('beauty') || 
                         categories.includes('skincare') ||
                         categories.includes('makeup') ||
                         categories.includes('hair') ||
                         categories.includes('perfume');
      
      return {
        found: true,
        source: 'Open Products Facts',
        productType: isCosmetic ? 'cosmetic' : 'food',
        product: {
          name: p.product_name || p.brands || 'Unknown Product',
          brand: p.brands || null,
          categories: p.categories || null,
          imageUrl: p.image_url || null,
          ingredients: p.ingredients_text || null,
        },
      };
    }
    return { found: false };
  } catch (error) {
    console.error('Open Products Facts error:', error);
    return { found: false };
  }
}

// Search the web for product information using Firecrawl
async function searchWebForProduct(barcode: string): Promise<string | null> {
  try {
    const firecrawlApiKey = Deno.env.get('FIRECRAWL_API_KEY');
    if (!firecrawlApiKey) {
      console.log('FIRECRAWL_API_KEY not configured, skipping web search');
      return null;
    }

    console.log(`Searching web for barcode: ${barcode}`);

    // Search for the product by barcode
    const response = await fetch('https://api.firecrawl.dev/v1/search', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${firecrawlApiKey}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        query: `"${barcode}" product ingredients review`,
        limit: 5,
        scrapeOptions: {
          formats: ['markdown'],
        },
      }),
    });

    if (!response.ok) {
      console.error('Firecrawl search error:', response.status, await response.text());
      return null;
    }

    const data = await response.json();
    
    if (!data.success || !data.data || data.data.length === 0) {
      console.log('No web search results found');
      return null;
    }

    // Combine search results into context
    const webContext = data.data.slice(0, 3).map((result: any) => {
      return `Source: ${result.url}\nTitle: ${result.title || 'N/A'}\nContent: ${result.markdown?.substring(0, 2000) || result.description || 'N/A'}`;
    }).join('\n\n---\n\n');

    console.log(`Found ${data.data.length} web search results`);
    return webContext;
  } catch (error) {
    console.error('Web search error:', error);
    return null;
  }
}

// AI-powered product analysis for cosmetics (with existing product info)
async function analyzeProductWithAI(barcode: string, product: ProductResult, productType: 'food' | 'cosmetic'): Promise<AIAnalysis | null> {
  try {
    const lovableApiKey = Deno.env.get('LOVABLE_API_KEY');
    if (!lovableApiKey) {
      console.log('LOVABLE_API_KEY not configured, skipping AI analysis');
      return null;
    }

    console.log(`Running AI analysis for ${productType}: ${product.name}`);

    // Search the web for additional product information
    const webContext = await searchWebForProduct(barcode);
    
    const webSearchInfo = webContext 
      ? `\n\nWEB SEARCH RESULTS:\n${webContext}\n\nUse the above web search results to provide accurate, up-to-date product analysis.`
      : '';

    const response = await fetch('https://ai.gateway.lovable.dev/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${lovableApiKey}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        model: 'google/gemini-3-flash-preview',
        messages: [
          {
            role: 'system',
            content: `You are a world-class cosmetic and skincare safety expert. Analyze products for safety, ingredient concerns, and provide helpful recommendations.

Your analysis should be thorough and research-backed. Focus on:
1. Ingredient safety and potential concerns
2. Skin type suitability
3. Known allergens or irritants
4. Benefits and positive aspects
5. Who should avoid this product
6. Safer alternatives if concerns exist

RESPONSE FORMAT - You MUST respond with valid JSON:
{
  "summary": "2-3 sentence overview of the product safety profile",
  "safetyRating": "Safe" | "Moderate Concern" | "High Concern",
  "keyIngredients": ["ingredient1", "ingredient2", "ingredient3"],
  "concerns": ["concern1 if any", "concern2 if any"],
  "benefits": ["benefit1", "benefit2"],
  "suitableFor": ["skin type or demographic"],
  "avoidIf": ["condition or sensitivity if any"],
  "alternatives": ["safer alternative product if concerns exist"],
  "sources": ["Brief mention of sources used for this analysis"]
}`
          },
          {
            role: 'user',
            content: `Analyze this ${productType} product for safety and suitability:

PRODUCT NAME: ${product.name}
BRAND: ${product.brand || 'Unknown'}
CATEGORIES: ${product.categories || 'Unknown'}
INGREDIENTS: ${product.ingredients || 'Not available'}${webSearchInfo}

Provide a comprehensive safety analysis based on the ingredients and any research findings.`
          }
        ],
        temperature: 0.3,
        max_completion_tokens: 1500,
      }),
    });

    if (!response.ok) {
      console.error('AI API error:', response.status, await response.text());
      return null;
    }

    const data = await response.json();
    const content = data.choices?.[0]?.message?.content;
    
    if (!content) {
      console.log('AI returned empty response');
      return null;
    }

    console.log('AI analysis response:', content);

    // Parse the JSON response
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      console.log('Could not extract JSON from AI response');
      return null;
    }

    const aiResult = JSON.parse(jsonMatch[0]);
    
    return {
      summary: aiResult.summary || '',
      safetyRating: aiResult.safetyRating || 'Moderate Concern',
      keyIngredients: aiResult.keyIngredients || [],
      concerns: aiResult.concerns || [],
      benefits: aiResult.benefits || [],
      suitableFor: aiResult.suitableFor || [],
      avoidIf: aiResult.avoidIf || [],
      alternatives: aiResult.alternatives || [],
      sources: aiResult.sources || [],
    };
  } catch (error) {
    console.error('AI analysis error:', error);
    return null;
  }
}

// AI-powered product identification and analysis using web search + OpenAI
async function lookupWithAI(barcode: string): Promise<LookupResult> {
  try {
    const lovableApiKey = Deno.env.get('LOVABLE_API_KEY');
    if (!lovableApiKey) {
      console.log('LOVABLE_API_KEY not configured, skipping AI lookup');
      return { found: false };
    }

    console.log(`Attempting comprehensive AI identification for barcode: ${barcode}`);

    // First, search the web for product information
    const webContext = await searchWebForProduct(barcode);
    
    const webSearchInfo = webContext 
      ? `\n\nWEB SEARCH RESULTS:\n${webContext}\n\nUse the above web search results to provide accurate, up-to-date product information and analysis.`
      : '';

    const response = await fetch('https://ai.gateway.lovable.dev/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${lovableApiKey}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        model: 'google/gemini-3-flash-preview',
        messages: [
          {
            role: 'system',
            content: `You are a world-class product research expert with deep knowledge of consumer products, cosmetics, food items, and their safety profiles. Your task is to identify products by barcode and provide comprehensive analysis based on web research.

When identifying a product, you MUST:
1. Use the barcode prefix to identify the brand
2. Analyze any web search results provided to get accurate product details
3. Provide the most accurate product match with detailed information
4. Include a thorough safety and ingredient analysis based on research

BRAND PREFIX DATABASE:
- 071249 = L'Oréal Paris (cosmetics - makeup, skincare, haircare)
- 192333 = Clinique (cosmetics - skincare, makeup, fragrances)
- 381370, 381519 = Neutrogena (skincare, sunscreen, haircare)
- 041167 = Banana Boat (sunscreen, sun care)
- 012044 = Old Spice (deodorant, body wash, cologne)
- 305210 = Olay (skincare, anti-aging)
- 074764 = Maybelline (makeup, mascara, lipstick)
- 041554 = Maybelline New York
- 309970 = Revlon (makeup, nail polish)
- 022700 = Dove (personal care, soap, deodorant)
- 037000 = Procter & Gamble brands
- 038000 = Kellogg's (cereals, snacks)
- 044000 = Nestlé (food, beverages)
- 049000 = Kraft (food products)
- 070470 = Colgate (oral care)
- 300410 = Garnier (haircare, skincare)
- 361422 = e.l.f. Cosmetics
- 609332 = CeraVe (skincare)
- 851954 = The Ordinary (skincare)

RESPONSE FORMAT - You MUST respond with valid JSON:
{
  "found": true,
  "name": "Full Product Name with Size/Variant",
  "brand": "Brand Name",
  "productType": "food" or "cosmetic",
  "categories": "Category1, Category2",
  "ingredients": "Full ingredient list if known, comma-separated",
  "analysis": {
    "summary": "2-3 sentence overview of the product based on research",
    "safetyRating": "Safe" | "Moderate Concern" | "High Concern",
    "keyIngredients": ["ingredient1", "ingredient2", "ingredient3"],
    "concerns": ["concern1 if any", "concern2 if any"],
    "benefits": ["benefit1", "benefit2"],
    "suitableFor": ["skin type or demographic"],
    "avoidIf": ["condition or sensitivity if any"],
    "alternatives": ["safer alternative product if concerns exist"],
    "sources": ["Brief mention of sources used for this analysis"]
  }
}

If you cannot identify the product at all, respond with: {"found": false}

IMPORTANT: Use the web search results to provide accurate, research-backed information. Cite findings from the search when relevant.`
          },
          {
            role: 'user',
            content: `Research and identify this product by barcode: ${barcode}${webSearchInfo}

Provide a comprehensive analysis including:
1. Product identification (name, brand, type) - use web search results if available
2. Full ingredient list from research
3. Safety assessment based on ingredient research and reviews
4. Any health concerns or benefits found in research
5. Who should use or avoid this product
6. Safer alternatives if there are concerns
7. Sources used for the analysis

Be thorough and cite the web research. The user is scanning this product to understand if it's safe for them.`
          }
        ],
        temperature: 0.3,
        max_completion_tokens: 2000,
      }),
    });

    if (!response.ok) {
      console.error('AI API error:', response.status, await response.text());
      return { found: false };
    }

    const data = await response.json();
    const content = data.choices?.[0]?.message?.content;
    
    if (!content) {
      console.log('AI returned empty response');
      return { found: false };
    }

    console.log('AI response:', content);

    // Parse the JSON response
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      console.log('Could not extract JSON from AI response');
      return { found: false };
    }

    const aiResult = JSON.parse(jsonMatch[0]);
    
    if (!aiResult.found || !aiResult.name) {
      console.log('AI could not identify the product');
      return { found: false };
    }

    console.log(`AI identified product: ${aiResult.name} by ${aiResult.brand}`);
    console.log(`AI analysis:`, aiResult.analysis);

    return {
      found: true,
      source: webContext ? 'AI + Web Research' : 'AI Research',
      productType: aiResult.productType === 'cosmetic' ? 'cosmetic' : 'food',
      aiGenerated: true,
      product: {
        name: aiResult.name,
        brand: aiResult.brand || null,
        categories: aiResult.categories || null,
        imageUrl: null,
        ingredients: aiResult.ingredients || null,
      },
      aiAnalysis: aiResult.analysis || null,
    };
  } catch (error) {
    console.error('AI lookup error:', error);
    return { found: false };
  }
}

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const { barcode, skipAI } = await req.json();

    if (!barcode) {
      return new Response(
        JSON.stringify({ error: 'Barcode is required' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log(`Looking up barcode: ${barcode}`);

    // First check our community database (fastest)
    const communityResult = await lookupCommunityProducts(barcode);
    if (communityResult.found) {
      return new Response(
        JSON.stringify({
          found: true,
          source: communityResult.source,
          productType: communityResult.productType,
          product: communityResult.product,
        }),
        { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Then try all Open*Facts databases in parallel
    const [foodResult, beautyResult, productsResult] = await Promise.all([
      lookupOpenFoodFacts(barcode),
      lookupOpenBeautyFacts(barcode),
      lookupOpenProductsFacts(barcode),
    ]);

    // Return first found result, prioritizing beauty for cosmetics
    let result: LookupResult = { found: false };
    
    if (beautyResult.found) {
      result = beautyResult;
    } else if (foodResult.found) {
      result = foodResult;
    } else if (productsResult.found) {
      result = productsResult;
    }

    // If not found in any database and AI is not skipped, try AI identification
    if (!result.found && !skipAI) {
      console.log('Product not found in databases, trying AI identification...');
      const aiResult = await lookupWithAI(barcode);
      if (aiResult.found) {
        result = aiResult;
      }
    }

    // Helper function to detect if a product is likely a cosmetic based on name/categories
    const isCosmeticProduct = (name: string, categories: string | null): boolean => {
      const cosmeticKeywords = [
        'lotion', 'لوشن', 'cream', 'كريم', 'shampoo', 'شامبو', 'conditioner', 
        'moisturizer', 'serum', 'sunscreen', 'makeup', 'mascara', 'lipstick',
        'foundation', 'concealer', 'perfume', 'cologne', 'deodorant', 'soap',
        'cleanser', 'toner', 'exfoliant', 'mask', 'scrub', 'gel', 'balm',
        'body wash', 'hand wash', 'face wash', 'skincare', 'haircare',
        'cosmetic', 'beauty', 'nail polish', 'eye shadow', 'blush', 'bronzer',
        'primer', 'setting spray', 'body butter', 'oil', 'essence', 'emulsion'
      ];
      const lowerName = name.toLowerCase();
      const lowerCategories = (categories || '').toLowerCase();
      return cosmeticKeywords.some(keyword => 
        lowerName.includes(keyword) || lowerCategories.includes(keyword)
      );
    };

    // Check if product found as food is actually a cosmetic (misclassified)
    if (result.found && result.productType === 'food' && result.product) {
      if (isCosmeticProduct(result.product.name, result.product.categories)) {
        console.log(`Product "${result.product.name}" appears to be a cosmetic, reclassifying...`);
        result.productType = 'cosmetic';
      }
    }

    // If cosmetic product found in database, enhance with AI analysis
    if (result.found && result.productType === 'cosmetic' && result.product && !skipAI) {
      console.log('Cosmetic found in database, running AI + Web Research analysis...');
      const aiAnalysis = await analyzeProductWithAI(barcode, result.product, 'cosmetic');
      if (aiAnalysis) {
        result.aiAnalysis = aiAnalysis;
        result.source = result.source + ' + AI Analysis';
      }
    }

    if (!result.found) {
      console.log(`Product not found in any database for barcode: ${barcode}`);
      return new Response(
        JSON.stringify({ found: false }),
        { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log(`Found product in ${result.source}: ${result.product?.name}`);

    return new Response(
      JSON.stringify({
        found: true,
        source: result.source,
        productType: result.productType,
        product: result.product,
        aiGenerated: result.aiGenerated || false,
        aiAnalysis: result.aiAnalysis || null,
      }),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error) {
    console.error('Error looking up barcode:', error);
    return new Response(
      JSON.stringify({ error: 'Failed to lookup barcode', found: false }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});