-- Create user health profiles table for TOBIN sensitivity thresholds
CREATE TABLE public.user_health_profiles (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL UNIQUE,
  -- Health conditions
  has_autoimmune boolean DEFAULT false,
  has_allergies boolean DEFAULT false,
  has_digestive_issues boolean DEFAULT false,
  has_hormonal_imbalance boolean DEFAULT false,
  has_cardiovascular_issues boolean DEFAULT false,
  has_skin_sensitivity boolean DEFAULT false,
  -- TOBIN category thresholds (1-10, lower = more sensitive)
  toxin_threshold integer DEFAULT 5 CHECK (toxin_threshold >= 1 AND toxin_threshold <= 10),
  oxidative_threshold integer DEFAULT 5 CHECK (oxidative_threshold >= 1 AND oxidative_threshold <= 10),
  biological_threshold integer DEFAULT 5 CHECK (biological_threshold >= 1 AND biological_threshold <= 10),
  inflammatory_threshold integer DEFAULT 5 CHECK (inflammatory_threshold >= 1 AND inflammatory_threshold <= 10),
  nutrient_threshold integer DEFAULT 5 CHECK (nutrient_threshold >= 1 AND nutrient_threshold <= 10),
  -- Overall threshold
  total_threshold integer DEFAULT 25 CHECK (total_threshold >= 5 AND total_threshold <= 50),
  -- Alerts preference
  show_alerts boolean DEFAULT true,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Enable RLS
ALTER TABLE public.user_health_profiles ENABLE ROW LEVEL SECURITY;

-- Create policies
CREATE POLICY "Users can view their own health profile"
ON public.user_health_profiles
FOR SELECT
USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own health profile"
ON public.user_health_profiles
FOR INSERT
WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own health profile"
ON public.user_health_profiles
FOR UPDATE
USING (auth.uid() = user_id);

-- Create trigger for updated_at
CREATE TRIGGER update_user_health_profiles_updated_at
BEFORE UPDATE ON public.user_health_profiles
FOR EACH ROW
EXECUTE FUNCTION public.update_updated_at_column();