-- Add group chat columns to conversations
ALTER TABLE public.conversations 
ADD COLUMN is_group boolean DEFAULT false,
ADD COLUMN name text,
ADD COLUMN created_by uuid;

-- Create conversation participants table for group chats
CREATE TABLE public.conversation_participants (
    id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
    conversation_id uuid REFERENCES public.conversations(id) ON DELETE CASCADE NOT NULL,
    user_id uuid NOT NULL,
    joined_at timestamp with time zone DEFAULT now() NOT NULL,
    left_at timestamp with time zone,
    UNIQUE (conversation_id, user_id)
);

-- Enable RLS
ALTER TABLE public.conversation_participants ENABLE ROW LEVEL SECURITY;

-- RLS policies for conversation_participants
CREATE POLICY "Users can view participants in their conversations"
ON public.conversation_participants
FOR SELECT
USING (
    EXISTS (
        SELECT 1 FROM public.conversation_participants cp
        WHERE cp.conversation_id = conversation_participants.conversation_id
        AND cp.user_id = auth.uid()
        AND cp.left_at IS NULL
    )
    OR EXISTS (
        SELECT 1 FROM public.conversations c
        WHERE c.id = conversation_participants.conversation_id
        AND (c.participant_1 = auth.uid() OR c.participant_2 = auth.uid())
    )
);

CREATE POLICY "Users can add participants to groups they created"
ON public.conversation_participants
FOR INSERT
WITH CHECK (
    EXISTS (
        SELECT 1 FROM public.conversations c
        WHERE c.id = conversation_id
        AND c.is_group = true
        AND c.created_by = auth.uid()
    )
    OR auth.uid() = user_id
);

CREATE POLICY "Users can leave conversations"
ON public.conversation_participants
FOR UPDATE
USING (user_id = auth.uid());

-- Update conversations policies for group chats
DROP POLICY IF EXISTS "Users can view their own conversations" ON public.conversations;
CREATE POLICY "Users can view their own conversations"
ON public.conversations
FOR SELECT
USING (
    participant_1 = auth.uid() 
    OR participant_2 = auth.uid()
    OR EXISTS (
        SELECT 1 FROM public.conversation_participants cp
        WHERE cp.conversation_id = id
        AND cp.user_id = auth.uid()
        AND cp.left_at IS NULL
    )
);

DROP POLICY IF EXISTS "Users can create conversations" ON public.conversations;
CREATE POLICY "Users can create conversations"
ON public.conversations
FOR INSERT
WITH CHECK (
    participant_1 = auth.uid() 
    OR participant_2 = auth.uid()
    OR created_by = auth.uid()
);

DROP POLICY IF EXISTS "Users can update their conversations" ON public.conversations;
CREATE POLICY "Users can update their conversations"
ON public.conversations
FOR UPDATE
USING (
    participant_1 = auth.uid() 
    OR participant_2 = auth.uid()
    OR EXISTS (
        SELECT 1 FROM public.conversation_participants cp
        WHERE cp.conversation_id = id
        AND cp.user_id = auth.uid()
        AND cp.left_at IS NULL
    )
);

-- Update messages policies for group chats
DROP POLICY IF EXISTS "Users can view messages in their conversations" ON public.messages;
CREATE POLICY "Users can view messages in their conversations"
ON public.messages
FOR SELECT
USING (
    EXISTS (
        SELECT 1 FROM public.conversations c
        WHERE c.id = messages.conversation_id
        AND (
            c.participant_1 = auth.uid() 
            OR c.participant_2 = auth.uid()
            OR EXISTS (
                SELECT 1 FROM public.conversation_participants cp
                WHERE cp.conversation_id = c.id
                AND cp.user_id = auth.uid()
                AND cp.left_at IS NULL
            )
        )
    )
);

DROP POLICY IF EXISTS "Users can send messages" ON public.messages;
CREATE POLICY "Users can send messages"
ON public.messages
FOR INSERT
WITH CHECK (
    auth.uid() = sender_id
    AND EXISTS (
        SELECT 1 FROM public.conversations c
        WHERE c.id = messages.conversation_id
        AND (
            c.participant_1 = auth.uid() 
            OR c.participant_2 = auth.uid()
            OR EXISTS (
                SELECT 1 FROM public.conversation_participants cp
                WHERE cp.conversation_id = c.id
                AND cp.user_id = auth.uid()
                AND cp.left_at IS NULL
            )
        )
    )
);

DROP POLICY IF EXISTS "Users can mark messages as read" ON public.messages;
CREATE POLICY "Users can mark messages as read"
ON public.messages
FOR UPDATE
USING (
    EXISTS (
        SELECT 1 FROM public.conversations c
        WHERE c.id = messages.conversation_id
        AND (
            c.participant_1 = auth.uid() 
            OR c.participant_2 = auth.uid()
            OR EXISTS (
                SELECT 1 FROM public.conversation_participants cp
                WHERE cp.conversation_id = c.id
                AND cp.user_id = auth.uid()
                AND cp.left_at IS NULL
            )
        )
    )
);